<?php
/**
 * ptyMysqlConfig
 *
 * MySQL 공통 설정 로더
 * 설정 파일: ~/.ptyMysqlConfig.ini
 */

namespace platyFramework;

/**
 * MySQL 설정 로더 클래스
 */
class ptyMysqlConfig
{
    private static $configPath = null;

    /**
     * 설정 파일 경로 반환
     */
    public static function getConfigPath()
    {
        if (self::$configPath === null) {
            self::$configPath = getenv('HOME') . '/.ptyMysqlConfig.ini';
        }
        return self::$configPath;
    }

    /**
     * 설정 파일에서 섹션 목록 조회
     */
    public static function getSections()
    {
        $configPath = self::getConfigPath();

        if (!file_exists($configPath)) {
            return array();
        }

        $config = parse_ini_file($configPath, true);
        return $config ? array_keys($config) : array();
    }

    /**
     * MySQL 설정 로드
     *
     * @param string $section INI 파일 섹션명 (기본값: default)
     * @return array 설정 배열 [host, username, password, database, charset]
     * @throws \Exception 설정 파일이나 섹션이 없을 경우
     */
    public static function load($section = 'default')
    {
        $configPath = self::getConfigPath();

        if (!file_exists($configPath)) {
            throw new \Exception("MySQL 설정 파일을 찾을 수 없습니다: {$configPath}\n\n" . self::getConfigExample());
        }

        $config = parse_ini_file($configPath, true);

        if ($config === false) {
            throw new \Exception("MySQL 설정 파일을 파싱할 수 없습니다: {$configPath}");
        }

        if (!isset($config[$section])) {
            $availableSections = implode(', ', array_keys($config));
            throw new \Exception("MySQL 설정에서 [{$section}] 섹션을 찾을 수 없습니다.\n사용 가능한 섹션: {$availableSections}");
        }

        $sectionConfig = $config[$section];

        // 필수 필드 검증
        if (!isset($sectionConfig['host']) || empty($sectionConfig['host'])) {
            throw new \Exception("MySQL 설정 [{$section}] 섹션에 필수 필드 'host'가 없습니다.");
        }

        return array(
            'host' => $sectionConfig['host'],
            'username' => isset($sectionConfig['username']) ? $sectionConfig['username'] : 'root',
            'password' => isset($sectionConfig['password']) ? trim($sectionConfig['password'], '"\'') : null,
            'database' => isset($sectionConfig['database']) ? $sectionConfig['database'] : null,
            'charset' => isset($sectionConfig['charset']) ? $sectionConfig['charset'] : 'utf8mb4',
        );
    }

    /**
     * mysqli 연결 생성
     *
     * @param string $section INI 파일 섹션명 (기본값: default)
     * @return array [connection => mysqli, config => array]
     * @throws \Exception 연결 실패 시
     */
    public static function connect($section = 'default')
    {
        $config = self::load($section);

        $connection = mysqli_connect(
            $config['host'],
            $config['username'],
            $config['password'],
            $config['database']
        );

        if (!$connection) {
            throw new \Exception("MySQL 연결 실패: " . mysqli_connect_error());
        }

        if (!empty($config['charset'])) {
            mysqli_set_charset($connection, $config['charset']);
        }

        return array(
            'connection' => $connection,
            'config' => $config,
        );
    }

    /**
     * 설정 파일 예시 반환
     */
    public static function getConfigExample()
    {
        return <<<EOT
설정 파일 예시 (~/.ptyMysqlConfig.ini):

[default]
host=localhost
username=root
password="your_password"
database=your_db
charset=utf8mb4

[production]
host=prod-mysql.example.com
username=admin
password="production_password"
database=prod_db
charset=utf8mb4
EOT;
    }
}
