#!/usr/bin/php
<?php
// CLI에서 실행하는 경우 인자로 MySQL 비밀번호를 받도록 처리
if (php_sapi_name() === "cli") {
    if ($argc < 2) {
        die("❌ 사용법: php " . $argv[0] . " <MySQL 비밀번호>\n");
    }
    $pass = $argv[1]; // 인자로 받은 MySQL 비밀번호
} else {
    die("❌ 이 스크립트는 CLI에서만 실행할 수 있습니다.\n");
}

$host = "localhost";
$user = "root";

// 현재 날짜에서 2달 전 날짜 구하기
$twoMonthsAgo = date('Y-m-d H:i:s', strtotime('-2 months'));

/**
 * 폴더의 총 용량을 계산하는 함수 (MB 단위)
 */
function getFolderSize($folderPath) {
    if (!is_dir($folderPath)) {
        return "❌ 없음";
    }
    $size = shell_exec("du -sb " . escapeshellarg($folderPath) . " | cut -f1");
    if ($size === null) {
        return "❌ 오류";
    }
    return round($size / (1024 * 1024), 2); // MB 단위로 변환
}

/**
 * CLI 표 출력 함수
 */
function printTable($data, $title) {
    global $colWidths;

    echo "\n$title\n";
    $divider = str_repeat("=", array_sum($colWidths) + count($colWidths) * 3);
    echo "$divider\n";

    // 헤더 출력
    printf(
        "| %-3s | %-25s | %-15s | %-15s | %-15s | %-14s | %-13s | %-13s | %-13s | %-20s | %-20s |\n",
        "상태", "데이터베이스", "테이블", "총 회원 수", "2달 이내 접속",
        "홈 폴더 존재", "/home 크기", "/var 크기", "총 크기", "가장 오래된 로그인", "최신 로그인"
    );

    echo "$divider\n"; // 헤더 아래 구분선 추가

    if (count($data) === 0) {
        echo "| 데이터 없음 |\n";
        echo "$divider\n";
        return;
    }

    // 데이터 출력
    foreach ($data as $row) {
        printf(
            "| %-3s | %-25s | %-15s | %15d | %15d | %-14s | %13s MB | %13s MB | %13s MB | %-20s | %-20s |\n",
            $row['status'], $row['database'], $row['table'],
            $row['totalUsers'], $row['activeUsers'],
            $row['homeFolder'], $row['homeFolderSize'], $row['mysqlFolderSize'], $row['totalSize'],
            $row['oldestLogin'], $row['recentLogin']
        );
    }
    echo "$divider\n";
}

try {
    // MySQL 서버에 연결
    $pdo = new PDO("mysql:host=$host", $user, $pass);
    $pdo->setAttribute(PDO::ATTR_ERRMODE, PDO::ERRMODE_EXCEPTION);

    // 컬럼 너비 설정
    $colWidths = array(
        "status" => 3,
        "database" => 25,
        "table" => 15,
        "totalUsers" => 15,
        "activeUsers" => 15,
        "homeFolder" => 14,
        "homeFolderSize" => 13,
        "mysqlFolderSize" => 13,
        "totalSize" => 13,
        "oldestLogin" => 20,
        "recentLogin" => 20
    );

    // 모든 데이터베이스 목록 가져오기
    $dbs = $pdo->query("SHOW DATABASES")->fetchAll(PDO::FETCH_COLUMN);

    $dbData = array();

    foreach ($dbs as $db) {
        $pdo->exec("USE `$db`");

        // 폴더 크기 계산
        $homeFolderPath = "/home/$db";
        $mysqlFolderPath = "/var/lib/mysql/$db";
        $folderExists = is_dir($homeFolderPath) ? "✅ 있음" : "❌ 없음";
        $homeFolderSize = getFolderSize($homeFolderPath);
        $mysqlFolderSize = getFolderSize($mysqlFolderPath);
        $totalSize = ($homeFolderSize > 0 ? $homeFolderSize : 0) + ($mysqlFolderSize > 0 ? $mysqlFolderSize : 0);

        // `t_pty_users`와 `t_user` 테이블이 존재하는지 확인
        $checkTables = array_merge(
            $pdo->query("SHOW TABLES LIKE 't_pty_users'")->fetchAll(PDO::FETCH_COLUMN),
            $pdo->query("SHOW TABLES LIKE 't_user'")->fetchAll(PDO::FETCH_COLUMN)
        );

        if (empty($checkTables)) continue;

        $tableName = in_array('t_pty_users', $checkTables) ? 't_pty_users' : 't_user';

        // 회원 수 및 2달 이내 접속한 회원 수 조회
        $stmtUsers = $pdo->query("SELECT COUNT(*) FROM `$tableName`");
        $totalUsers = $stmtUsers->fetchColumn();

        $stmtActiveUsers = $pdo->prepare("SELECT COUNT(*) FROM `$tableName` WHERE lastLoginDateTime > :date");
        $stmtActiveUsers->execute(array('date' => $twoMonthsAgo));
        $activeUsers = $stmtActiveUsers->fetchColumn();

        // 로그인 날짜 조회
        $stmt = $pdo->query("SELECT MIN(lastLoginDateTime) AS minLogin, MAX(lastLoginDateTime) AS maxLogin FROM `$tableName`");
        $loginData = $stmt->fetch(PDO::FETCH_ASSOC);
        if (!$loginData || !$loginData['minLogin']) continue;

        $status = ($loginData['maxLogin'] <= $twoMonthsAgo) ? "🔴" : "🟢";

        $dbData[] = array(
            'status' => $status,
            'database' => $db,
            'table' => $tableName,
            'totalUsers' => $totalUsers,
            'activeUsers' => $activeUsers,
            'homeFolder' => $folderExists,
            'homeFolderSize' => $homeFolderSize,
            'mysqlFolderSize' => $mysqlFolderSize,
            'totalSize' => $totalSize,
            'oldestLogin' => $loginData['minLogin'],
            'recentLogin' => $loginData['maxLogin']
        );
    }

    // 총 크기 기준으로 내림차순 정렬
    usort($dbData, function ($a, $b) {
        if ($b['totalSize'] > $a['totalSize']) return 1;
        if ($b['totalSize'] < $a['totalSize']) return -1;
        return 0;
    });

    // 데이터 구분
    $oldData = array();
    $recentData = array();

    foreach ($dbData as $data) {
        if ($data['status'] === "🔴") {
            $oldData[] = $data;
        } else {
            $recentData[] = $data;
        }
    }

    // 출력 실행
    printTable($oldData, "🔴 2달 이상 지난 로그인 DB 목록 (크기 정렬)");
    printTable($recentData, "🟢 2달 이내 로그인한 DB 목록 (크기 정렬)");

} catch (PDOException $e) {
    echo "❌ 에러 발생: " . $e->getMessage();
}
?>

