# CLAUDE.md

이 파일은 Claude Code가 이 저장소의 코드 작업 시 참고하는 가이드입니다.

## 프로젝트 개요

pty_centos.git은 Elasticsearch, MySQL 등 다양한 서비스를 다루는 CLI 도구 모음입니다.

## 네임스페이스

모든 PHP 코드는 `platyFramework` 네임스페이스를 사용합니다.

## ptyElastic* 스크립트 작성 규칙

새로운 Elasticsearch CLI 스크립트 생성 시 다음 패턴을 따릅니다:

### 필수 구조

```php
#!/usr/bin/env php
<?php
/**
 * 스크립트 설명
 *
 * 설정 파일: ~/.ptyElasticConfig.ini
 *
 * Usage: ./ptyElasticXXX <args> [options]
 */

namespace platyFramework;

require_once __DIR__ . '/ptyLibrary_PHP/cli/ptyCliOptionParser.php';
require_once __DIR__ . '/ptyLibrary_PHP/elastic/ptyElasticConfig.php';

// 인자 파싱
$parsed = ptyCliOptionParser::parse($argv);
$positionalArgs = $parsed['positional'];
$options = $parsed['options'];
$elasticSection = $options['elastic'] ?? 'default';
$verbose = isset($options['verbose']);

// 도움말 또는 필수 인자 확인
if (/* 필수 인자 누락 */ || isset($options['help'])) {
    echo "사용법: {$argv[0]} <필수인자> [옵션]\n";
    echo "\n";
    echo "옵션:\n";
    echo "  --elastic=섹션명    INI 파일 섹션 (기본값: default)\n";
    echo "  --verbose           상세 로그 출력\n";
    echo "  --help              도움말 출력\n";
    echo "\n";
    // ... 추가 도움말 ...
    echo "설정 파일: ~/.ptyElasticConfig.ini\n";
    echo ptyElasticConfig::getConfigExample() . "\n";
    exit(isset($options['help']) ? 0 : 1);
}

try {
    // Elasticsearch 연결
    $connection = ptyElasticConfig::connect($elasticSection);
    $elastic = $connection['client'];
    $elastic->setDebug($verbose);  // --verbose 옵션에 따라 로그 제어
    $config = $connection['config'];
    $authMethod = $connection['authMethod'];

    // ... 비즈니스 로직 ...

} catch (\Exception $e) {
    echo "Error: " . $e->getMessage() . "\n";
    exit(1);
}
```

### 필수 옵션

| 옵션 | 설명 |
|------|------|
| `--elastic=섹션명` | INI 파일 섹션 지정 (기본값: default) |
| `--verbose` | 상세 로그 출력 (기본: 비활성화) |
| `--help` | 도움말 출력 |

### 주요 클래스/함수

- `ptyCliOptionParser::parse($argv)` - CLI 인자 파싱 (positional, options 분리)
- `ptyElasticConfig::connect($section)` - Elasticsearch 연결 및 클라이언트 반환
- `ptyElasticConfig::getConfigExample()` - 설정 파일 예시 문자열 반환
- `$elastic->setDebug($verbose)` - 상세 로그 활성화/비활성화

### Exit 코드

- `0`: 정상 종료 (--help 포함)
- `1`: 에러 또는 필수 인자 누락

## ptyCliLog 사용법

`ptyCliLog`는 **인스턴스 메소드**를 사용하는 클래스입니다. 정적 메소드가 아니므로 반드시 인스턴스를 생성해서 사용해야 합니다.

### 올바른 사용법

```php
// 헬퍼 함수로 인스턴스 생성
$log = ptyCliLog("APP", ptyCliLog::COLOR_CYAN, true);

// 인스턴스 메소드 호출
$log->info("정보 메시지");
$log->success("성공 메시지");
$log->warning("경고 메시지");
$log->error("에러 메시지");
```

### 잘못된 사용법 (금지)

```php
// ❌ 정적 메소드처럼 호출하면 에러 발생
ptyCliLog::info("메시지");  // Fatal error!
```

### 파라미터

- `$prefix`: 로그 prefix (예: "APP", "RUN", "CRON")
- `$color`: 로그 색상 (예: `ptyCliLog::COLOR_CYAN`)
- `$debug`: 디버그 모드 (기본값: true)

## ptyLibrary_PHP 구조

```
ptyLibrary_PHP/
├── cli/
│   ├── ptyCliOptionParser.php  # CLI 옵션 파서
│   ├── ptyCliLog.php           # 컬러 로깅 (인스턴스 메소드)
│   └── ptyCliColor.php         # ANSI 색상 코드
├── elastic/
│   ├── ptyElasticConfig.php    # Elasticsearch 설정 로더
│   └── Elastic.php             # Elasticsearch 클라이언트
└── ...
```

## 설정 파일

### ~/.ptyElasticConfig.ini

```ini
[default]
host=https://localhost:9200
apiKey=your_api_key

# 또는 user/password 방식
# user=elastic
# password="your_password"

[production]
host=https://prod-elastic:9200
apiKey=production_api_key
```

### ~/.ptyMysqlConfig.ini

```ini
[default]
host=localhost
username=root
password="your_password"
database=your_db
charset=utf8mb4
```

## 유틸리티 스크립트

### ptyRun - 스크립트 중복 실행 제어

스크립트의 중복 실행을 제어하고 타임아웃을 관리하는 래퍼 스크립트입니다. 모든 stdout/stderr 출력을 실시간으로 표시하고 로그 파일에 저장합니다.

```bash
# 기본 사용 (중복 실행 차단, 로그는 ./ptyRun_logs/에 90일 보관)
ptyRun "php /path/to/script.php"

# 중복 실행 허용
ptyRun "php script.php" --run-duplicate=true

# 최대 실행 시간 설정 (60분)
ptyRun "php script.php" --max-run-time=60

# 로그 저장 안 함
ptyRun "php script.php" --no-log

# 로그 7일만 보관
ptyRun "php script.php" --log-rotate-date=7

# 로그 10개만 보관 + 커스텀 로그 폴더
ptyRun "php script.php" --log-rotate-count=10 --log-dir=/var/log/myapp
```

**옵션:**
- `--run-duplicate=true|false`: 중복 실행 허용 여부 (기본값: **false**)
- `--max-run-time=분`: 최대 실행 시간(분), 초과 시 강제 종료 후 재실행 (기본값: 0 - 무제한)
- `--no-log`: 로그 파일 저장 비활성화 (기본값: 로그 저장 **활성화**)
- `--log-dir=경로`: 로그 저장 폴더 (기본값: 스크립트폴더/ptyRun_logs)
- `--log-rotate-date=일수`: 로그 보관 일수, 초과 시 오래된 로그 삭제 (기본값: **90**)
- `--log-rotate-count=개수`: 로그 보관 개수, 초과 시 오래된 로그 삭제 (기본값: 0 - 무제한)

**로그 파일:**
- 파일명 형식: `스크립트명_YYYYMMDD_HHMMSS.log`
- 저장 위치: 기본값은 ptyRun 스크립트가 있는 폴더의 `ptyRun_logs/`
- stdout/stderr 모두 기록 (stderr는 `[STDERR]` 프리픽스 추가)

### ptyCronBuild - 크론탭 자동 등록

스크립트를 크론탭에 자동으로 등록하는 유틸리티입니다.

```bash
# 5분마다 실행
ptyCronBuild "ptyRun 'php /path/to/script.php' --max-run-time=60" --daily-5min

# 매일 0시 실행
ptyCronBuild "php /path/to/backup.php" --daily

# 1시간마다 실행
ptyCronBuild "php script.php" --daily-hour

# 10분마다 실행하되, 10분 내에 한 번만 실행 (중복 실행 차단 + 최대 10분 타임아웃)
# ptyRun의 기본 동작(--run-duplicate=false)으로 10분 내에 이미 실행 중이면 스킵
# --max-run-time=600으로 10분 초과 시 강제 종료 후 재실행
ptyCronBuild 'ptyRun "cd /home/redmine/git; php git_update.php" --max-run-time=600' --daily-10min
```

**스케줄 옵션:**
- `--daily`: 매일 0시에 실행
- `--daily-5min`: 매일 5분마다 실행
- `--daily-10min`: 매일 10분마다 실행
- `--daily-30min`: 매일 30분마다 실행
- `--daily-hour`: 매일 1시간마다 실행

**ptyRun과 함께 사용 시:**
- `ptyRun`의 기본값(`--run-duplicate=false`)으로 중복 실행이 자동 차단됨
- `--max-run-time`으로 최대 실행 시간 설정 시, 해당 시간 초과 시 프로세스를 강제 종료하고 재실행
- 예: `--daily-10min`과 `--max-run-time=600`을 함께 사용하면 10분마다 크론이 실행되지만, 이전 실행이 아직 진행 중이면 스킵하고, 10분 이상 실행 중이면 강제 종료 후 재실행
