#!/bin/bash

#######################################
# Git 설정 관리 스크립트
# 사용법: ./pty_git.sh
#######################################

set -e  # 에러 발생 시 스크립트 중단

# 색상 정의
CYAN='\033[0;36m'
BLUE='\033[1;94m'  # 밝은 파란색
PURPLE='\033[0;35m'
GREEN='\033[0;32m'
YELLOW='\033[1;33m'
RED='\033[0;31m'
NC='\033[0m' # No Color

# 로고 출력
clear
echo -e "${CYAN}"
cat << "EOF"

░█▀█░█░░░█▀█░▀█▀░█░█░░░█░█░█▀█░█░█░█▀▀░█▀▀
░█▀▀░█░░░█▀█░░█░░░█░░░░█▀█░█░█░█░█░▀▀█░█▀▀
░▀░░░▀▀▀░▀░▀░░▀░░░▀░░░░▀░▀░▀▀▀░▀▀▀░▀▀▀░▀▀▀
EOF
echo -e "${NC}"
echo -e "${PURPLE}═══════════════════════════════════════════════════════════${NC}"
echo ""
echo -e "${BLUE} Git 설정 관리 스크립트 v1.0 (pty_git.sh)${NC}"
echo ""
echo -e "${BLUE} Copyright (C) 2019 by cpueblo@platyhouse.com, PlatyHouse Co.,LTD.${NC}"
echo -e "${BLUE} Visit https://www.platyhouse.com${NC}"
echo ""
#echo -e "${PURPLE}═══════════════════════════════════════════════════════════${NC}"
#echo ""

# 메뉴 출력 함수
show_menu() {
    echo -e "${PURPLE}═══════════════════════════════════════════════════════════${NC}"
    echo ""
    echo -e "  ${CYAN}1.${NC} Config 조회"
    echo -e "  ${CYAN}2.${NC} 계정 정보 조회 및 변경"
    echo -e "  ${CYAN}3.${NC} 계정 정보 삭제"
    echo -e "  ${CYAN}4.${NC} 인증 정보 저장 설정"
    echo -e "  ${CYAN}5.${NC} 인증 정보 조회"
    echo -e "  ${CYAN}q.${NC} 종료"
    echo ""
    echo -e "${YELLOW}선택:${NC} \c"
}

#######################################
# 1. Config 조회
#######################################
view_config() {
    echo -e "${CYAN}[1]${NC} Git Config 조회"
    echo ""
    echo -e "${BLUE}현재 Git 설정 목록:${NC}"
    echo -e "${PURPLE}═══════════════════════════════════════════════════════════${NC}"
    git config --list
    echo -e "${PURPLE}═══════════════════════════════════════════════════════════${NC}"
    echo ""
}

#######################################
# 2. 계정 정보 조회 및 변경
#######################################
manage_account() {
    echo -e "${CYAN}[2]${NC} 계정 정보 조회 및 변경"
    echo ""

    # 현재 설정 값 조회
    CURRENT_NAME=$(git config --global user.name 2>/dev/null || echo "설정되지 않음")
    CURRENT_EMAIL=$(git config --global user.email 2>/dev/null || echo "설정되지 않음")

    echo -e "${BLUE}현재 설정된 계정 정보:${NC}"
    echo -e "  이름 (user.name): ${YELLOW}${CURRENT_NAME}${NC}"
    echo -e "  이메일 (user.email): ${YELLOW}${CURRENT_EMAIL}${NC}"
    echo ""

    echo -e "${YELLOW}계정 정보를 변경하시겠습니까? (y/N):${NC} \c"
    read -n 1 -r REPLY_CHANGE
    echo

    if [[ $REPLY_CHANGE =~ ^[Yy]$ ]]; then
        echo ""
        echo -e "${YELLOW}새로운 이름을 입력하세요 (엔터: 건너뛰기):${NC} \c"
        read NEW_NAME

        echo -e "${YELLOW}새로운 이메일을 입력하세요 (엔터: 건너뛰기):${NC} \c"
        read NEW_EMAIL

        if [ -n "$NEW_NAME" ]; then
            git config --global user.name "$NEW_NAME"
            echo -e "  ${GREEN}✓${NC} 이름이 설정되었습니다: ${YELLOW}${NEW_NAME}${NC}"
        fi

        if [ -n "$NEW_EMAIL" ]; then
            git config --global user.email "$NEW_EMAIL"
            echo -e "  ${GREEN}✓${NC} 이메일이 설정되었습니다: ${YELLOW}${NEW_EMAIL}${NC}"
        fi

        if [ -z "$NEW_NAME" ] && [ -z "$NEW_EMAIL" ]; then
            echo -e "  ${YELLOW}⊘${NC} 변경사항이 없습니다."
        fi
    else
        echo -e "  ${YELLOW}⊘${NC} 계정 정보 변경을 건너뜁니다."
    fi
    echo ""
}

#######################################
# 3. 계정 정보 삭제
#######################################
delete_account() {
    echo -e "${CYAN}[3]${NC} 계정 정보 삭제"
    echo ""

    # 현재 설정 값 조회
    CURRENT_NAME=$(git config --global user.name 2>/dev/null || echo "설정되지 않음")
    CURRENT_EMAIL=$(git config --global user.email 2>/dev/null || echo "설정되지 않음")

    echo -e "${BLUE}현재 설정된 계정 정보:${NC}"
    echo -e "  이름 (user.name): ${YELLOW}${CURRENT_NAME}${NC}"
    echo -e "  이메일 (user.email): ${YELLOW}${CURRENT_EMAIL}${NC}"
    echo ""

    echo -e "${RED}경고: 계정 정보를 삭제하시겠습니까? (y/N):${NC} \c"
    read -n 1 -r REPLY_DELETE
    echo

    if [[ $REPLY_DELETE =~ ^[Yy]$ ]]; then
        echo ""
        git config --global --unset user.name 2>/dev/null && echo -e "  ${GREEN}✓${NC} user.name이 삭제되었습니다." || echo -e "  ${YELLOW}⊘${NC} user.name이 설정되어 있지 않습니다."
        git config --global --unset user.email 2>/dev/null && echo -e "  ${GREEN}✓${NC} user.email이 삭제되었습니다." || echo -e "  ${YELLOW}⊘${NC} user.email이 설정되어 있지 않습니다."
    else
        echo -e "  ${YELLOW}⊘${NC} 계정 정보 삭제를 취소했습니다."
    fi
    echo ""
}

#######################################
# 4. 인증 정보 저장 설정
#######################################
setup_credential() {
    echo -e "${CYAN}[4]${NC} 인증 정보 저장 설정"
    echo ""

    # 현재 설정 값 조회
    CURRENT_HELPER=$(git config --global credential.helper 2>/dev/null || echo "설정되지 않음")

    echo -e "${BLUE}현재 credential helper 설정:${NC}"
    echo -e "  ${YELLOW}${CURRENT_HELPER}${NC}"
    echo ""

    echo -e "${YELLOW}인증 정보 저장 방식을 선택하세요:${NC}"
    echo -e "  ${CYAN}1.${NC} 디스크에 저장 (store) - 평문으로 저장됨"
    echo -e "  ${CYAN}2.${NC} 메모리에 저장 (cache) - 15분간 유지"
    echo -e "  ${CYAN}3.${NC} 메모리에 저장 (cache) - 1시간 유지"
    echo -e "  ${CYAN}4.${NC} 메모리에 저장 (cache) - 사용자 지정 시간"
    echo -e "  ${CYAN}5.${NC} 인증 정보 저장 해제"
    echo -e "  ${CYAN}6.${NC} 건너뛰기"
    echo -e "${YELLOW}선택 (1/2/3/4/5/6):${NC} \c"
    read -n 1 -r REPLY_CREDENTIAL
    echo
    echo

    case $REPLY_CREDENTIAL in
        1)
            git config --global credential.helper store
            echo -e "  ${GREEN}✓${NC} 인증 정보를 디스크에 저장하도록 설정되었습니다."
            echo -e "  ${YELLOW}[참고]${NC} 인증 정보는 ~/.git-credentials 파일에 평문으로 저장됩니다."
            ;;
        2)
            git config --global credential.helper cache
            echo -e "  ${GREEN}✓${NC} 인증 정보를 메모리에 15분간 저장하도록 설정되었습니다."
            ;;
        3)
            git config --global credential.helper 'cache --timeout=3600'
            echo -e "  ${GREEN}✓${NC} 인증 정보를 메모리에 1시간 동안 저장하도록 설정되었습니다."
            ;;
        4)
            echo -e "${YELLOW}저장 시간을 초 단위로 입력하세요 (예: 7200 = 2시간):${NC} \c"
            read TIMEOUT
            if [[ "$TIMEOUT" =~ ^[0-9]+$ ]]; then
                git config --global credential.helper "cache --timeout=${TIMEOUT}"
                MINUTES=$((TIMEOUT / 60))
                echo -e "  ${GREEN}✓${NC} 인증 정보를 메모리에 ${MINUTES}분 동안 저장하도록 설정되었습니다."
            else
                echo -e "  ${RED}✗${NC} 잘못된 입력입니다. 숫자만 입력해주세요."
            fi
            ;;
        5)
            git config --global --unset credential.helper 2>/dev/null && echo -e "  ${GREEN}✓${NC} 인증 정보 저장 설정이 해제되었습니다." || echo -e "  ${YELLOW}⊘${NC} 설정되어 있지 않습니다."
            ;;
        6)
            echo -e "  ${YELLOW}⊘${NC} 인증 정보 저장 설정을 건너뜁니다."
            ;;
        *)
            echo -e "  ${RED}잘못된 선택입니다.${NC}"
            ;;
    esac
    echo ""
}

#######################################
# 5. 인증 정보 조회
#######################################
view_credential() {
    echo -e "${CYAN}[5]${NC} 인증 정보 조회"
    echo ""

    # 현재 credential helper 설정 조회
    CURRENT_HELPER=$(git config --global credential.helper 2>/dev/null || echo "설정되지 않음")

    echo -e "${BLUE}현재 credential.helper 설정:${NC}"
    echo -e "  ${YELLOW}${CURRENT_HELPER}${NC}"
    echo ""

    # ~/.git-credentials 파일 확인
    CRED_FILE="$HOME/.git-credentials"

    if [ -f "$CRED_FILE" ]; then
        echo -e "${BLUE}저장된 인증 정보 (~/.git-credentials):${NC}"
        echo -e "${PURPLE}═══════════════════════════════════════════════════════════${NC}"

        # 파일 내용을 읽어서 비밀번호 부분을 마스킹
        while IFS= read -r line; do
            # URL 패턴에서 비밀번호 부분을 ****로 마스킹
            if [[ $line =~ ^https?://([^:]+):([^@]+)@(.+)$ ]]; then
                username="${BASH_REMATCH[1]}"
                password="${BASH_REMATCH[2]}"
                host="${BASH_REMATCH[3]}"
                masked_password=$(echo "$password" | sed 's/./*/g')
                echo -e "  ${GREEN}https://${username}:${masked_password}@${host}${NC}"
            else
                echo -e "  ${GREEN}${line}${NC}"
            fi
        done < "$CRED_FILE"

        echo -e "${PURPLE}═══════════════════════════════════════════════════════════${NC}"
        echo ""
        echo -e "${YELLOW}[참고]${NC} 비밀번호는 보안을 위해 *로 표시됩니다."
        echo -e "${YELLOW}[경고]${NC} 실제 파일에는 평문으로 저장되어 있습니다: ${CRED_FILE}"
    else
        echo -e "${YELLOW}저장된 인증 정보 파일이 없습니다.${NC}"
        echo -e "  파일 위치: ${CRED_FILE}"
    fi
    echo ""

    # 추가 credential 관련 설정 표시
    echo -e "${BLUE}기타 credential 관련 설정:${NC}"
    git config --global --get-regexp credential\. 2>/dev/null || echo -e "  ${YELLOW}추가 설정 없음${NC}"
    echo ""
}

#######################################
# 메인 실행 로직
#######################################

# 메뉴 반복
while true; do
    # 메뉴 표시
    show_menu

    # 사용자 입력 받기
    read MENU_CHOICE

    echo ""

    # 종료 체크
    if [[ "$MENU_CHOICE" == "q" ]] || [[ "$MENU_CHOICE" == "Q" ]]; then
        echo -e "${YELLOW}프로그램을 종료합니다.${NC}"
        break
    fi

    # 선택된 항목 실행
    case $MENU_CHOICE in
        1)
            view_config
            ;;
        2)
            manage_account
            ;;
        3)
            delete_account
            ;;
        4)
            setup_credential
            ;;
        5)
            view_credential
            ;;
        *)
            echo -e "${RED}잘못된 선택: ${MENU_CHOICE}${NC}"
            echo ""
            ;;
    esac
done

echo ""
echo -e "${PURPLE}═══════════════════════════════════════════════════════════${NC}"
echo -e "${GREEN}✓ Git 설정 관리가 완료되었습니다!${NC}"
echo -e "${PURPLE}═══════════════════════════════════════════════════════════${NC}"
echo ""
