#!/usr/bin/env php
<?php
/**
 * ptyMysqlRestore
 *
 * MySQL 데이터베이스 복원 도구
 * 설정 파일: ~/.ptyMysqlConfig.ini
 *
 * Usage: ./ptyMysqlRestore <pattern> [options]
 *        ./ptyMysqlRestore "*"           (현재 경로의 모든 sql 파일)
 *        ./ptyMysqlRestore "mydb.*.sql"  (mydb의 모든 테이블)
 */

namespace platyFramework;

require_once __DIR__ . '/ptyLibrary_PHP/cli/ptyCliOptionParser.php';
require_once __DIR__ . '/ptyLibrary_PHP/mysql/ptyMysqlConfig.php';

// 인자 파싱
$parsed = ptyCliOptionParser::parse($argv);
$positionalArgs = $parsed['positional'];
$options = $parsed['options'];
$mysqlSection = $options['mysql'] ?? 'default';
$verbose = isset($options['verbose']);
$force = isset($options['force']);
$inputDir = $options['input'] ?? '.';
$targetDatabase = $options['database'] ?? null;

// 도움말 또는 필수 인자 확인
if (empty($positionalArgs) || isset($options['help'])) {
    echo "사용법: {$argv[0]} <pattern> [옵션]\n";
    echo "\n";
    echo "인자:\n";
    echo "  <pattern>           파일 패턴 (* = 전체, mydb.*.sql = 특정 DB)\n";
    echo "\n";
    echo "옵션:\n";
    echo "  --mysql=섹션명      INI 파일 섹션 (기본값: default)\n";
    echo "  --database=DB명     복원할 대상 데이터베이스 (미지정시 파일명에서 추출)\n";
    echo "  --input=경로        입력 디렉토리 (기본값: 현재 디렉토리)\n";
    echo "  --force             확인 없이 바로 복원 실행\n";
    echo "  --verbose           상세 로그 출력\n";
    echo "  --help              도움말 출력\n";
    echo "\n";
    echo "예시:\n";
    echo "  {$argv[0]} \"*\"                          # 모든 sql 파일 복원\n";
    echo "  {$argv[0]} \"*.sql\"                      # 모든 sql 파일 복원\n";
    echo "  {$argv[0]} \"mydb.*.sql\"                 # mydb의 모든 테이블 복원\n";
    echo "  {$argv[0]} \"mydb.users.sql\"             # 특정 파일 복원\n";
    echo "  {$argv[0]} \"*\" --input=/backup          # /backup 디렉토리에서 복원\n";
    echo "  {$argv[0]} \"*\" --mysql=production       # production 섹션 사용\n";
    echo "  {$argv[0]} \"*\" --force                  # 확인 없이 복원\n";
    echo "  {$argv[0]} \"*\" --database=mydb_dev      # 모든 파일을 mydb_dev DB에 복원\n";
    echo "\n";
    echo "파일명 형식: <database>.<table>.sql\n";
    echo "\n";
    echo "설정 파일: ~/.ptyMysqlConfig.ini\n";
    echo ptyMysqlConfig::getConfigExample() . "\n";
    exit(isset($options['help']) ? 0 : 1);
}

$pattern = $positionalArgs[0];

// 패턴 정규화
if ($pattern === '*') {
    $pattern = '*.sql';
}
if (!preg_match('/\.sql$/i', $pattern)) {
    $pattern .= '.sql';
}

// ANSI 색상 코드
$RED = "\033[1;31m";
$GREEN = "\033[1;32m";
$YELLOW = "\033[1;33m";
$CYAN = "\033[1;36m";
$MAGENTA = "\033[1;35m";
$RESET = "\033[0m";

// 타임스탬프 로그 함수
function logMessage($message, $verbose = false, $isVerbose = false) {
    if ($isVerbose && !$verbose) {
        return;
    }
    $timestamp = date('Y-m-d H:i:s');
    echo "[$timestamp] $message\n";
}

// 바이트를 읽기 쉬운 형식으로 변환
function formatBytes($bytes) {
    if ($bytes >= 1073741824) {
        return number_format($bytes / 1073741824, 2) . ' GB';
    } elseif ($bytes >= 1048576) {
        return number_format($bytes / 1048576, 2) . ' MB';
    } elseif ($bytes >= 1024) {
        return number_format($bytes / 1024, 2) . ' KB';
    } else {
        return $bytes . ' bytes';
    }
}

// 파일명에서 DB명, 테이블명 추출 (mydb.tablename.sql -> [mydb, tablename])
function extractDbAndTable($filename) {
    $basename = basename($filename);
    $parts = explode('.', $basename);
    if (count($parts) >= 3) {
        $dbName = $parts[0];
        // 마지막 .sql 제거하고 나머지를 테이블명으로
        array_pop($parts); // .sql 제거
        array_shift($parts); // DB명 제거
        $tableName = implode('.', $parts);
        return [$dbName, $tableName];
    }
    return [null, null];
}

// 복원 실행 함수
function restoreFile($host, $user, $password, $dbName, $sqlFile, $verbose) {
    $basename = basename($sqlFile);
    $size = filesize($sqlFile);
    $sizeStr = formatBytes($size);

    // mysql 명령 생성
    $cmd = "mysql";
    $cmd .= " -h " . escapeshellarg($host);
    $cmd .= " -u " . escapeshellarg($user);
    $cmd .= " -p" . escapeshellarg($password);
    $cmd .= " --default-character-set=utf8mb4";
    $cmd .= " " . escapeshellarg($dbName);
    $cmd .= " < " . escapeshellarg($sqlFile);
    $cmd .= " 2>&1";

    if ($verbose) {
        logMessage("명령: $cmd", $verbose, true);
    }

    exec($cmd, $output, $returnCode);

    if ($returnCode !== 0) {
        logMessage("ERROR: $basename 복원 실패", true);
        if (!empty($output)) {
            logMessage(implode("\n", $output), true);
        }
        return false;
    }

    logMessage("OK: $basename ($sizeStr)", true);
    return true;
}

try {
    // MySQL 연결 (DB 목록 조회용)
    $conn = ptyMysqlConfig::connect($mysqlSection);
    $connection = $conn['connection'];
    $config = $conn['config'];

    // 입력 디렉토리 확인
    if (!is_dir($inputDir)) {
        throw new \Exception("입력 디렉토리가 없습니다: $inputDir");
    }

    // 패턴에 맞는 파일 찾기
    $searchPattern = rtrim($inputDir, '/') . '/' . $pattern;
    $files = glob($searchPattern);

    if (empty($files)) {
        logMessage("패턴에 맞는 파일이 없습니다: $pattern", true);
        exit(0);
    }

    sort($files); // 파일명 순서로 정렬

    // ============================================
    // 복원 정보 표시
    // ============================================
    echo "\n";
    echo "{$CYAN}╔══════════════════════════════════════════════════════════════════╗{$RESET}\n";
    echo "{$CYAN}║                    MySQL 복원 정보 확인                          ║{$RESET}\n";
    echo "{$CYAN}╚══════════════════════════════════════════════════════════════════╝{$RESET}\n";
    echo "\n";

    // 서버 정보
    echo "{$YELLOW}[ 서버 정보 ]{$RESET}\n";
    echo str_repeat("-", 50) . "\n";
    echo "  Host     : {$config['host']}\n";
    echo "  User     : {$config['username']}\n";
    echo "  Section  : $mysqlSection\n";

    // MySQL 버전 조회
    $versionResult = mysqli_query($connection, "SELECT VERSION() as version");
    if ($versionResult && $row = mysqli_fetch_assoc($versionResult)) {
        echo "  Version  : {$row['version']}\n";
    }

    // 대상 DB 지정시 표시
    if ($targetDatabase) {
        echo "  {$CYAN}Target DB: $targetDatabase{$RESET}\n";
    }
    echo "\n";

    // 복원 대상 파일 분석
    $restoreInfo = [];
    $totalSize = 0;
    $dbList = [];

    foreach ($files as $sqlFile) {
        list($origDbName, $tableName) = extractDbAndTable($sqlFile);

        if (!$origDbName) {
            continue;
        }

        // --database 옵션이 있으면 해당 DB로, 없으면 파일명에서 추출한 DB 사용
        $dbName = $targetDatabase ?? $origDbName;

        $size = filesize($sqlFile);
        $totalSize += $size;

        if (!isset($dbList[$dbName])) {
            $dbList[$dbName] = [];
        }
        $dbList[$dbName][] = [
            'table' => $tableName,
            'file' => $sqlFile,
            'size' => $size,
            'origDb' => $origDbName,
        ];
    }

    // DB별 복원 대상 표시
    echo "{$YELLOW}[ 복원 대상 ]{$RESET}\n";
    echo str_repeat("-", 50) . "\n";
    echo "  Input    : " . realpath($inputDir) . "\n";
    echo "  Pattern  : $pattern\n";
    echo "  Files    : " . count($files) . "개\n";
    echo "  Total    : " . formatBytes($totalSize) . "\n";
    echo "\n";

    foreach ($dbList as $dbName => $tables) {
        // DB 존재 여부 확인
        $checkDb = mysqli_query($connection, "SHOW DATABASES LIKE '$dbName'");
        $dbExists = mysqli_num_rows($checkDb) > 0;
        $dbStatus = $dbExists ? "{$GREEN}[EXISTS]{$RESET}" : "{$YELLOW}[NEW]{$RESET}";

        echo "{$CYAN}  Database: $dbName{$RESET} $dbStatus\n";

        foreach ($tables as $info) {
            $tableName = $info['table'];
            $sizeStr = formatBytes($info['size']);

            // 원본 DB가 다르면 표시
            $origInfo = "";
            if ($targetDatabase && $info['origDb'] !== $dbName) {
                $origInfo = " {$MAGENTA}← {$info['origDb']}{$RESET}";
            }

            // 테이블 존재 여부 확인
            $tableStatus = "";
            if ($dbExists) {
                $checkTable = mysqli_query($connection, "SHOW TABLES FROM `$dbName` LIKE '$tableName'");
                if ($checkTable && mysqli_num_rows($checkTable) > 0) {
                    $tableStatus = "{$RED}[OVERWRITE]{$RESET}";
                } else {
                    $tableStatus = "{$GREEN}[NEW]{$RESET}";
                }
            } else {
                $tableStatus = "{$GREEN}[NEW]{$RESET}";
            }

            echo "    - $tableName ($sizeStr) $tableStatus$origInfo\n";
        }
        echo "\n";
    }

    // 경고 메시지
    echo "{$RED}╔══════════════════════════════════════════════════════════════════╗{$RESET}\n";
    echo "{$RED}║  주의: [OVERWRITE] 표시된 테이블은 기존 데이터가 삭제됩니다!    ║{$RESET}\n";
    echo "{$RED}╚══════════════════════════════════════════════════════════════════╝{$RESET}\n";
    echo "\n";

    // 확인 절차 (--force가 없을 때만)
    if (!$force) {
        echo "복원을 진행하시겠습니까? (y/N): ";
        $handle = fopen("php://stdin", "r");
        $line = fgets($handle);
        fclose($handle);

        $answer = strtolower(trim($line));
        if ($answer !== 'y' && $answer !== 'yes') {
            echo "\n복원이 취소되었습니다.\n";
            mysqli_close($connection);
            exit(0);
        }
        echo "\n";
    }

    // ============================================
    // 복원 실행
    // ============================================
    echo str_repeat("=", 60) . "\n";
    logMessage("MySQL 복원 시작", true);
    echo str_repeat("=", 60) . "\n";

    $totalRestores = 0;
    $successRestores = 0;

    foreach ($dbList as $dbName => $tables) {
        // DB 존재 확인 및 생성
        $checkDb = mysqli_query($connection, "SHOW DATABASES LIKE '$dbName'");
        if (mysqli_num_rows($checkDb) === 0) {
            logMessage("DB 생성: $dbName", true);
            mysqli_query($connection, "CREATE DATABASE IF NOT EXISTS `$dbName` DEFAULT CHARACTER SET utf8mb4 COLLATE utf8mb4_unicode_ci");
        }

        foreach ($tables as $info) {
            $totalRestores++;
            if (restoreFile($config['host'], $config['username'], $config['password'], $dbName, $info['file'], $verbose)) {
                $successRestores++;
            }
        }
    }

    mysqli_close($connection);

    echo "\n" . str_repeat("=", 60) . "\n";
    logMessage("복원 완료: $successRestores / $totalRestores", true);

    if ($successRestores < $totalRestores) {
        exit(1);
    }

} catch (\Exception $e) {
    echo "Error: " . $e->getMessage() . "\n";
    exit(1);
}
