#!/bin/bash

# pty_files_old_files.sh
# 오래된 파일을 찾아서 목록을 보여주는 스크립트
# 사용법:
#   ./pty_files_old_files.sh -d 10    # 10일 이상 지난 파일 검색
#   ./pty_files_old_files.sh -h 24    # 24시간 이상 지난 파일 검색

show_usage() {
    echo "사용법: $0 [-d 일수] [-h 시간] [검색경로]"
    echo ""
    echo "옵션:"
    echo "  -d 일수      지정한 일수보다 오래된 파일 검색"
    echo "  -h 시간      지정한 시간보다 오래된 파일 검색"
    echo "  -p 경로      검색할 경로 (기본값: 현재 디렉토리)"
    echo "  -t 타입      파일 타입 (f: 파일만, d: 디렉토리만, 기본값: 모두)"
    echo "  -s          파일 크기와 함께 표시"
    echo "  -l          상세 정보 표시 (ls -lh 형식)"
    echo ""
    echo "예제:"
    echo "  $0 -d 10              # 현재 디렉토리에서 10일 이상 지난 파일 검색"
    echo "  $0 -h 24              # 현재 디렉토리에서 24시간 이상 지난 파일 검색"
    echo "  $0 -d 7 -p /var/log   # /var/log에서 7일 이상 지난 파일 검색"
    echo "  $0 -d 30 -t f -s      # 30일 이상 지난 파일만 크기와 함께 검색"
    exit 1
}

# 기본값 설정
DAYS=""
HOURS=""
SEARCH_PATH="."
FILE_TYPE=""
SHOW_SIZE=false
SHOW_DETAIL=false

# 옵션 파싱
while getopts "d:h:p:t:sl" opt; do
    case $opt in
        d)
            DAYS="$OPTARG"
            ;;
        h)
            HOURS="$OPTARG"
            ;;
        p)
            SEARCH_PATH="$OPTARG"
            ;;
        t)
            FILE_TYPE="$OPTARG"
            ;;
        s)
            SHOW_SIZE=true
            ;;
        l)
            SHOW_DETAIL=true
            ;;
        *)
            show_usage
            ;;
    esac
done

# -d 또는 -h 옵션이 반드시 필요
if [ -z "$DAYS" ] && [ -z "$HOURS" ]; then
    echo "오류: -d (일수) 또는 -h (시간) 옵션이 필요합니다."
    echo ""
    show_usage
fi

# 둘 다 지정된 경우 에러
if [ -n "$DAYS" ] && [ -n "$HOURS" ]; then
    echo "오류: -d와 -h 옵션은 동시에 사용할 수 없습니다."
    echo ""
    show_usage
fi

# 숫자 유효성 검사
if [ -n "$DAYS" ]; then
    if ! [[ "$DAYS" =~ ^[0-9]+$ ]]; then
        echo "오류: 일수는 숫자여야 합니다."
        exit 1
    fi
fi

if [ -n "$HOURS" ]; then
    if ! [[ "$HOURS" =~ ^[0-9]+$ ]]; then
        echo "오류: 시간은 숫자여야 합니다."
        exit 1
    fi
fi

# 검색 경로 확인
if [ ! -d "$SEARCH_PATH" ]; then
    echo "오류: 검색 경로 '$SEARCH_PATH'가 존재하지 않습니다."
    exit 1
fi

# find 명령어 옵션 구성
FIND_CMD="find \"$SEARCH_PATH\""

# 파일 타입 옵션
if [ "$FILE_TYPE" = "f" ]; then
    FIND_CMD="$FIND_CMD -type f"
elif [ "$FILE_TYPE" = "d" ]; then
    FIND_CMD="$FIND_CMD -type d"
fi

# 시간 옵션
if [ -n "$DAYS" ]; then
    FIND_CMD="$FIND_CMD -mtime +$DAYS"
    TIME_DESC="${DAYS}일"
else
    # 시간을 분으로 변환 (24시간 = 1440분)
    MINUTES=$((HOURS * 60))
    FIND_CMD="$FIND_CMD -mmin +$MINUTES"
    TIME_DESC="${HOURS}시간"
fi

# 헤더 출력
echo "============================================"
echo "오래된 파일 검색 결과"
echo "============================================"
echo "검색 경로: $SEARCH_PATH"
echo "검색 기준: ${TIME_DESC} 이상 지난 파일"
if [ -n "$FILE_TYPE" ]; then
    if [ "$FILE_TYPE" = "f" ]; then
        echo "파일 타입: 파일만"
    else
        echo "파일 타입: 디렉토리만"
    fi
fi
echo "============================================"
echo ""

# 파일 찾기 및 결과 출력
if [ "$SHOW_DETAIL" = true ]; then
    # 상세 정보 표시
    eval $FIND_CMD -exec ls -lhd {} \; 2>/dev/null | sort -k9
    COUNT=$(eval $FIND_CMD 2>/dev/null | wc -l)
elif [ "$SHOW_SIZE" = true ]; then
    # 크기와 함께 표시
    eval $FIND_CMD -exec ls -lhd {} \; 2>/dev/null | awk '{print $5 "\t" $9}' | sort -k2
    COUNT=$(eval $FIND_CMD 2>/dev/null | wc -l)
else
    # 기본: 경로만 표시
    eval $FIND_CMD 2>/dev/null | sort
    COUNT=$(eval $FIND_CMD 2>/dev/null | wc -l)
fi

# 결과 요약
echo ""
echo "============================================"
echo "총 ${COUNT}개의 항목을 찾았습니다."
echo "============================================"

# 총 크기 계산 (파일인 경우만)
if [ "$FILE_TYPE" = "f" ] || [ -z "$FILE_TYPE" ]; then
    TOTAL_SIZE=$(eval $FIND_CMD -type f -exec du -cb {} + 2>/dev/null | tail -1 | cut -f1)
    if [ -n "$TOTAL_SIZE" ]; then
        # 바이트를 사람이 읽기 쉬운 형식으로 변환
        TOTAL_SIZE_HUMAN=$(numfmt --to=iec-i --suffix=B $TOTAL_SIZE 2>/dev/null || echo "${TOTAL_SIZE} bytes")
        echo "총 파일 크기: $TOTAL_SIZE_HUMAN"
        echo "============================================"
    fi
fi

exit 0
