#!/bin/bash
#######################################
# ptyUtils 전역 설치 스크립트
#
# 설치 경로: /var/lib/ptyUtils.git
# 설정 파일: /etc/bashrc
#
# 사용법:
#   curl -sSL https://gitlab.platyhouse.com/platyhouse/centos/-/raw/master/ptyInstallGlobal.sh | bash
#
# 또는:
#   wget -qO- https://gitlab.platyhouse.com/platyhouse/centos/-/raw/master/ptyInstallGlobal.sh | bash
#
#######################################

set -e  # 에러 발생 시 스크립트 중단

# 색상 정의
CYAN='\033[0;36m'
GREEN='\033[0;32m'
YELLOW='\033[1;33m'
RED='\033[0;31m'
NC='\033[0m' # No Color

# 설치 경로
INSTALL_DIR="/var/lib/ptyUtils.git"
GIT_URL="https://gitlab.platyhouse.com/platyhouse/centos.git"

echo ""
echo -e "${CYAN}╔══════════════════════════════════════════════════════════════╗${NC}"
echo -e "${CYAN}║          ptyUtils 전역 설치 스크립트 (Global)                ║${NC}"
echo -e "${CYAN}╚══════════════════════════════════════════════════════════════╝${NC}"
echo ""

# root 권한 확인
if [[ $EUID -ne 0 ]]; then
    echo -e "${YELLOW}[INFO]${NC} root 권한이 필요합니다. sudo로 재실행합니다..."
    exec sudo bash "$0" "$@"
fi

# git 설치 확인
if ! command -v git &> /dev/null; then
    echo -e "${YELLOW}[INFO]${NC} git이 설치되어 있지 않습니다. 설치 중..."

    # OS 감지 및 git 설치
    if command -v yum &> /dev/null; then
        yum install -y git
    elif command -v dnf &> /dev/null; then
        dnf install -y git
    elif command -v apt-get &> /dev/null; then
        apt-get update && apt-get install -y git
    else
        echo -e "${RED}[ERROR]${NC} git을 자동 설치할 수 없습니다. 수동으로 설치해주세요."
        exit 1
    fi

    echo -e "${GREEN}[OK]${NC} git 설치 완료"
fi

# 기존 설치 확인
if [[ -d "$INSTALL_DIR" ]]; then
    echo -e "${YELLOW}[INFO]${NC} 기존 설치가 발견되었습니다: $INSTALL_DIR"
    echo -e "${YELLOW}[INFO]${NC} git pull로 업데이트합니다..."

    cd "$INSTALL_DIR"
    git pull origin master

    echo -e "${GREEN}[OK]${NC} 업데이트 완료"
else
    echo -e "${YELLOW}[INFO]${NC} 신규 설치를 진행합니다..."
    echo -e "${YELLOW}[INFO]${NC} 경로: $INSTALL_DIR"

    # 부모 디렉토리 생성
    mkdir -p "$(dirname "$INSTALL_DIR")"

    # git clone
    git clone "$GIT_URL" "$INSTALL_DIR"

    echo -e "${GREEN}[OK]${NC} 설치 완료"
fi

# 실행 권한 부여
echo -e "${YELLOW}[INFO]${NC} 실행 권한 설정 중..."
chmod +x "$INSTALL_DIR"/pty*

echo -e "${GREEN}[OK]${NC} 실행 권한 설정 완료"

# ptySetBash 실행 (--install-global 옵션)
echo ""
echo -e "${CYAN}══════════════════════════════════════════════════════════════${NC}"
echo -e "${CYAN} ptySetBash --install-global 실행${NC}"
echo -e "${CYAN}══════════════════════════════════════════════════════════════${NC}"
echo ""

cd "$INSTALL_DIR"
bash ./ptySetBash --install-global

echo ""
echo -e "${GREEN}╔══════════════════════════════════════════════════════════════╗${NC}"
echo -e "${GREEN}║          ptyUtils 전역 설치가 완료되었습니다!                ║${NC}"
echo -e "${GREEN}╚══════════════════════════════════════════════════════════════╝${NC}"
echo ""
echo -e "설치 경로: ${CYAN}$INSTALL_DIR${NC}"
echo -e "설정 파일: ${CYAN}/etc/bashrc${NC}"
echo -e "변경사항 적용: ${YELLOW}source /etc/bashrc${NC} 또는 새 터미널 열기"
echo ""
